<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreCategoryRequest;
use App\Http\Requests\UpdateCategoryRequest;
use App\Models\Category;
use Illuminate\Support\Str;

class CategoryController extends Controller
{
    public function __construct()
    {
        view()->share('type', request('type'));
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $type = request('type');
        $relationMap = [
            'package' => 'packages',
            'blog' => 'blogs',
            'faq' => 'faqs',
        ];

        $query = Category::query()
            ->withCount('children')
            ->orderBy('name');

        if (isset($relationMap[$type])) {
            $query->withCount($relationMap[$type]);
        }

        $categories = $query->get();

        return view('admin.category.index', compact('categories', 'relationMap'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $category = new Category;

        return view('admin.category.create', compact('category'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreCategoryRequest $request, $type)
    {
        $data = $request->validated();
        $data['status'] = $request->has('status');
        $data['slug'] = Str::slug($data['name']);
        $data['type'] = $type;

        try {
            $category = Category::updateOrCreate([
                'slug' => $data['slug'],
            ], $data);

            if ($request->has('thumbnail')) {
                // $data['thumbnail'] = $request->file('thumbnail')->store('categories', 'public');
                $category->image()->create([
                    'path' => $request->file('thumbnail')->store($type.'/categories', 'public'),
                    'type' => 'thumbnail',
                ]);
            }

            return redirect()->route('admin.category.index', $type)->with('success', 'Category created successfully.');
        } catch (\Exception $e) {
            return $e->getMessage();

            return redirect()->back()->with('error', 'Category could not be created.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($type, Category $category)
    {
        abort(404);

        return view('admin.category.show', compact('category'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($type, Category $category)
    {
        return view('admin.category.create', compact('category'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateCategoryRequest $request, $type, Category $category)
    {
        $data = $request->validated();
        $data['status'] = $request->has('status');
        $data['slug'] = Str::slug($data['name']);
        if ($request->has('thumbnail')) {
            $path = public_path('storage/'.$category->thumbnail);
            if (file_exists($path)) {
                @unlink($path);
            }
            $data['thumbnail'] = $request->file('thumbnail')->store('categories', 'public');
        }

        try {
            return redirect()->route('admin.category.index', $type)->with('success', 'Category updated successfully.');
        } catch (\Exception $e) {
            return $e->getMessage();

            return redirect()->back()->with('error', 'Category could not be updated.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($type, Category $category)
    {
        try {
            $path = public_path('storage/'.$category->thumbnail);
            $category->delete();
            if (file_exists($path)) {
                @unlink($path);
            }

            return response()->json(['message' => 'Category deleted successfully.', 'status' => 'success']);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Category could not be deleted.', 'status' => 'error'], 500);
        }
    }
}
